/*
@Time : 2020/8/20 下午8:19
@Author : xiukang
@File : main.go
@Software: GoLand
*/

package main

import (
	"github.com/gorilla/websocket"
	"net/http"
	"swasocket/controllers/message"
	userApi "swasocket/controllers/user"
	"swasocket/model"
	"swasocket/pkg/zaplogger"
	"swasocket/routers"
	"swasocket/runtime"
	"swasocket/server"
	"swasocket/server/user"
)

var upgrader = websocket.Upgrader{

	CheckOrigin: func(r *http.Request) bool {
		return true
	},
}

func init() {
	runtime.InitLog()
}

// 用户退出
func logout(msg model.LoginMessage, c *websocket.Conn) {
	if msg.FromId != "" {
		userServer := user.NewUserServer()
		userServer.LoginOut(msg)
	}
	err := c.Close()
	if err != nil {
		zaplogger.Info("disconnect:", msg, err)
		// 再关闭一次
		err := c.Close()
		if err != nil {
			zaplogger.Info("disconnect second:", msg, err)
		}
	}

	return
}

// 用户登陆
func login(loginMsg model.LoginMessage, c *websocket.Conn) (code int16, res string) {
	zaplogger.Info("收到消息连接消息：->", loginMsg)
	if loginMsg.Method != "User/Connect" {
		return 1, "参数有误或未登陆"
	}
	userServer := user.NewUserServer()
	return userServer.Connect(loginMsg, c)
}

// ws消息处理
func onWsMessage(w http.ResponseWriter, r *http.Request) {
	var loginMsg model.LoginMessage

	c, err := upgrader.Upgrade(w, r, nil)
	if err != nil {
		//zaplogger.Error(err)
		c.Close()
		return
	}

	err = c.ReadJSON(&loginMsg)
	if err != nil {
		//zaplogger.Error(err)
		c.Close()
		return
	}

	// 关闭连接需要修改
	defer logout(loginMsg, c)

	if loginMsg.FromId == "" {
		return
	}

	// 1.验证域名黑名单
	domain := loginMsg.Data["SiteRoot"]
	if domain == nil || domain == "" {
		zaplogger.Info("domain is null,uid:", loginMsg.FromId)
		return
	}

	baseServer := server.NewBaseServer()

	code, res := baseServer.CheckDomainWhite(domain.(string))
	if code > 0 {
		zaplogger.Info("check domain white error:"+res+" ("+c.RemoteAddr().String()+")", domain.(string), "userId:", loginMsg.FromId)
		return
	}

	// 2.接收到用户连接,执行登录
	code, res = login(loginMsg, c)
	if code > 0 {
		zaplogger.Error("connect error:"+res, code, loginMsg)
		return
	}

	// 系统监听用户消息
	for {
		// 1.处理当前用户获取系统消息
		var userMsg model.ReceiveMessage
		err = c.ReadJSON(&userMsg)
		if err != nil {
			zaplogger.Error("收到消息 json解析err:", err)
			break
			//if strings.Contains(fmt.Sprint(err), "websocket: close") {
			//	break
			//}
		}
		if userMsg.Method == "User/Connect" {
			continue
		}

		// 处理心跳
		if userMsg.Method == "Message/Heartbeat" {
			var returnData map[string]interface{}
			returnData = make(map[string]interface{})
			returnData["message"] = userMsg.Message
			returnData["fromId"] = userMsg.FromId
			userMsg.Data = returnData
			userMsg.Type = 1
			err := c.WriteJSON(userMsg)
			if err != nil {
				zaplogger.Error("发送心跳err:", err)
			}

			continue
		}

		if (userMsg.Method != "Message/Heartbeat") && (userMsg.Message == "") {
			continue
		}

		// TODO message 去掉 client 和指针
		zaplogger.Info("收到消息：->", userMsg)

		//model.MessageBroadcast <- userMsg
		code, res := routers.GetRouter(userMsg)
		zaplogger.Info("处理结果：->", code, res)
	}

}

/**
处理用户消息
*/
func handleMessages() {
	for {
		// 获取到管道里的所有数据 code为系统处理业务逻辑失败需要记录的问题
		msg := <-model.MessageBroadcast
		code, res := routers.GetRouter(msg)
		if code != 0 {
			zaplogger.Error(res)
			model.ErrorMessage = append(model.ErrorMessage, msg)
		}
	}
}

func main() {

	httpRouter := routers.NewMuxHandler()
	// 需要手动维护

	// 添加http路由
	httpRouter.Handle("/api/message/sendMessageToUser", &message.SendMessageToUser{})
	// 添加http路由
	httpRouter.Handle("/api/message/addDomainWhite", &message.AddDomainWhite{})
	// 添加http路由
	httpRouter.Handle("/api/user/getOnlineUsers", &userApi.GetOnlineUsers{})

	// 添加ws路由
	httpRouter.HandleFunc("/wss", onWsMessage)

	// 处理用户消息
	//go handleMessages()
	zaplogger.Info("websocket start at 127.0.0.1:3001")
	err := http.ListenAndServe(":3001", httpRouter)
	if err != nil {
		panic("ListenAndServe: " + err.Error())
	}
}
