package zaplogger

import (
	"errors"
	"os"
	"time"

	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
	"gopkg.in/natefinch/lumberjack.v2"
)

// error zapLogger
var (
	sugaredLogger       *zap.SugaredLogger
	SugaredLoggerNoSkip *zap.SugaredLogger
	RawLogger           *zap.Logger

	loglevel = &level{zapcore.InfoLevel}
)

var timeFormat = "2006-01-02 15:04:05.999999999 -0700 MST"

// 是否是trace级别的输出
var isTraceLevel = false

type level struct {
	lv zapcore.Level
}

func (l *level) Enabled(lv zapcore.Level) bool {
	return l.lv.Enabled(lv)
}

func (l *level) Set(lv zapcore.Level) {
	l.lv = lv
}

var levelMap = map[string]zapcore.Level{
	"trace":  zapcore.DebugLevel,
	"debug":  zapcore.DebugLevel,
	"info":   zapcore.InfoLevel,
	"warn":   zapcore.WarnLevel,
	"error":  zapcore.ErrorLevel,
	"dpanic": zapcore.DPanicLevel,
	"panic":  zapcore.PanicLevel,
	"fatal":  zapcore.FatalLevel,
}

type LogConfig struct {
	Encoding   string
	Filename   string
	MaxSize    int
	MaxAge     int
	MaxBackups int
	LocalTime  bool
	Compress   bool
	Level      string
	TimeFormat string
}

var Logc LogConfig

var logLevel = [6]string{"trace", "debug", "info", "warn", "error", "fatal"}

func GetLoggerLevel(index int) (level string, err error) {
	if index+1 >= len(logLevel) {
		return "", errors.New("log index error")
	}

	level = logLevel[index+1]
	return level, nil
}

func getLoggerLevel(lvl string) zapcore.Level {
	if level, ok := levelMap[lvl]; ok {
		return level
	}
	return zapcore.InfoLevel
}

func init() {
	logc := LogConfig{
		Filename:   "./zap.log", // 日志文件路径
		MaxSize:    1,           // megabytes(兆字节)
		MaxBackups: 7,           // 最多保留3个备份
		MaxAge:     3,           // days
		Compress:   true,        // 是否压缩 disabled by default
		LocalTime:  true,        // 用于格式化备份文件中的时间戳的时间是计算机的本地时间。 默认是使用UTC
	}
	_ = InitLogger("error", logc)
}

func InitLogger(level string, logc LogConfig) error {
	if logc.Filename == "" {
		return errors.New("invalid LogConfig")
	}
	Logc = logc
	log := lumberjack.Logger{
		Filename:   logc.Filename,   // 日志文件路径
		MaxSize:    logc.MaxSize,    // megabytes(兆字节)
		MaxBackups: logc.MaxBackups, // 最多保留3个备份
		MaxAge:     logc.MaxAge,     // days
		Compress:   logc.Compress,   // 是否压缩 disabled by default
		LocalTime:  logc.LocalTime,  // 用于格式化备份文件中的时间戳的时间是计算机的本地时间。 默认是使用UTC
	}

	lv := getLoggerLevel(level)
	loglevel.Set(lv)
	if "trace" == level {
		isTraceLevel = true
	}

	syncWriter := zapcore.AddSync(&log)
	timeEncoder := func(t time.Time, enc zapcore.PrimitiveArrayEncoder) {
		if logc.TimeFormat != "" {
			enc.AppendString(t.Format(logc.TimeFormat))
		} else {
			enc.AppendString(t.Format(timeFormat))
		}
	}
	encoder := zapcore.EncoderConfig{
		//Keys can be anything except the empty string.
		TimeKey:        "time",                       // json时时间键
		LevelKey:       "level",                      // json时日志等级键
		NameKey:        "name",                       // json时日志记录器键
		CallerKey:      "caller",                     // json时日志文件信息键
		MessageKey:     "message",                    // json时日志消息键
		StacktraceKey:  "stack",                      // json时堆栈键
		LineEnding:     zapcore.DefaultLineEnding,    // 友好日志换行符
		EncodeLevel:    zapcore.CapitalLevelEncoder,  // 友好日志等级名大小写（info INFO）
		EncodeTime:     timeEncoder,                  // 友好日志时日期格式化
		EncodeDuration: zapcore.NanosDurationEncoder, // 时间序列化
		EncodeCaller:   zapcore.ShortCallerEncoder,   // 日志文件信息（包/文件.go:行号）
	}

	core := zapcore.NewTee(
		// 友好的格式、输出控制台、动态等级
		//zapcore.NewCore(zapcore.NewConsoleEncoder(encoder), os.Stdout, lv),
		zapcore.NewCore(newKvEncoder(encoder), os.Stdout, loglevel),

		// 友好的格式、输出文件、处定义等级规则
		zapcore.NewCore(encoderByConfig(logc.Encoding, encoder), syncWriter, loglevel),
	)
	logger := zap.New(core, zap.AddCaller(), zap.AddCallerSkip(1))
	sugaredLogger = logger.Sugar()
	SugaredLoggerNoSkip = zap.New(core, zap.AddCaller()).Sugar()

	RawLogger = zap.New(core)

	return nil
}

func UpdateLogger(level string) {
	lv := getLoggerLevel(level)
	loglevel.Set(lv)

	return
}

func encoderByConfig(enc string, encConfig zapcore.EncoderConfig) zapcore.Encoder {
	switch enc {
	case "json":
		return zapcore.NewJSONEncoder(encConfig)
	case "console":
		return zapcore.NewConsoleEncoder(encConfig)
	case "kv":
		return newKvEncoder(encConfig)
	}

	return zapcore.NewConsoleEncoder(encConfig)
}

// TimeEncoder  格式化时间
func TimeEncoder(t time.Time, enc zapcore.PrimitiveArrayEncoder) {
	enc.AppendString(t.Format(timeFormat))
}

// Trace trace输出
func Trace(args ...interface{}) bool {
	if isTraceLevel {
		sugaredLogger.Debug(args...)
	}
	return isTraceLevel
}

// Tracef Tracef输出
func Tracef(template string, args ...interface{}) bool {
	if isTraceLevel {
		sugaredLogger.Debugf(template, args...)
	}
	return isTraceLevel
}

// Debug  debug输出
func Debug(args ...interface{}) {
	sugaredLogger.Debug(args...)
}

// Debugf Debug输出
func Debugf(template string, args ...interface{}) {
	sugaredLogger.Debugf(template, args...)
}

// Info Info输出
func Info(args ...interface{}) {
	sugaredLogger.Info(args...)
}

// Infof Trace
func Infof(template string, args ...interface{}) {
	sugaredLogger.Infof(template, args...)
}

// Warn warm输出
func Warn(args ...interface{}) {
	sugaredLogger.Warn(args...)
}

// Warnf Warn输出
func Warnf(template string, args ...interface{}) {
	sugaredLogger.Warnf(template, args...)
}

// Error error输出
func Error(args ...interface{}) {
	sugaredLogger.Error(args...)
}

// Errorf error输出
func Errorf(template string, args ...interface{}) {
	sugaredLogger.Errorf(template, args...)
}

// DPanic DPanic输出
func DPanic(args ...interface{}) {
	sugaredLogger.DPanic(args...)
}

// DPanicf DPanic输出
func DPanicf(template string, args ...interface{}) {
	sugaredLogger.DPanicf(template, args...)
}

// Panic Panic输出
func Panic(args ...interface{}) {
	sugaredLogger.Panic(args...)
}

// Panicf Panicf输出
func Panicf(template string, args ...interface{}) {
	sugaredLogger.Panicf(template, args...)
}

// Fatal Fatal输出
func Fatal(args ...interface{}) {
	sugaredLogger.Fatal(args...)
}

// Fatalf Fatalf输出
func Fatalf(template string, args ...interface{}) {
	sugaredLogger.Fatalf(template, args...)
}

// Logger 给IRIS装载自定义的Log库
type Logger struct {
}

// Debug Debug
func (l *Logger) Debug(args ...interface{}) {
	sugaredLogger.Debug(args...)
}

// Info Info
func (l *Logger) Info(args ...interface{}) {
	sugaredLogger.Info(args...)
}

// Warn Warn
func (l *Logger) Warn(args ...interface{}) {
	sugaredLogger.Warn(args...)
}

// Error Error
func (l *Logger) Error(args ...interface{}) {
	sugaredLogger.Error(args...)
}

// Println Println
func (l *Logger) Println(args ...interface{}) {
	sugaredLogger.Debug(args...)
}

// Print Print
func (l *Logger) Print(args ...interface{}) {
	sugaredLogger.Debug(args...)
}

func DefaultLogger() *zap.SugaredLogger {
	return sugaredLogger
}
