package server

import (
	"encoding/json"
	"fmt"
	"github.com/gorilla/websocket"
	"io/ioutil"
	"net/http"
	"os"
	"reflect"
	"strings"
	"swasocket/config"
	"swasocket/model"
	"swasocket/pkg/zaplogger"
)

type BaseServer struct {
}

func NewBaseServer() *BaseServer {
	return &BaseServer{}
}

type checkDomain struct {
	Message  string `json:"message"`
	Redirect string `json:"redirect"`
	Type     string `json:"type"`
}

// @Title CheckDomainWhite
// @Description 验证域名白名单
// @Param   domain  string   域名
// @return   code  int16  返回码
// @return   message  string   消息

func (b *BaseServer) CheckDomainWhite(domain string) (code int16, msg string) {
	return 0, "ok"
	//path := config.ApiHost + "?i=4&c=entry&m=swa_supersale&do=api&r=authorize.query&siteroot=" + domain
	//resp, err := http.Get(path)
	//return 0,"成功"
	// 1.域名写死
	//domain1 := strings.Contains(domain, ".wayxq.")
	//domain2 := strings.Contains(domain, ".fu505.")
	//
	//code = 1
	//if domain1 || domain2 {
	//	code = 0
	//}
	//
	//return code, msg

	// 2.调接口获取域名
	path := config.ApiHost + "?i=4&c=entry&m=swa_supersale&do=api&r=authorize.query&siteroot=" + domain
	resp, err := http.Get(path)
	if err != nil {
		return 1, "验证接口地址失败"
	}
	defer resp.Body.Close()
	body, _ := ioutil.ReadAll(resp.Body)

	var res checkDomain
	_ = json.Unmarshal(body, &res)

	if res.Type != "success" {
		code = 1
	}
	return code, res.Message

	//// 3.获取文件
	//msg = "成功"
	//res := b.CheckDomain(domain)
	//if !res {
	//	code = 1
	//	msg = "失败"
	//}
	//return
}

// @Description:  验证域名是否在配置文件里
// @receiver b
// @param host string
// @return bool

func (b *BaseServer) CheckDomain(host string) bool {
	domainStr, err := b.ReadFile(config.DomainPath)
	if err != nil {
		return false
	}

	var domain model.Domain
	err = json.Unmarshal([]byte(domainStr), &domain)
	if err != nil {
		return false
	}

	for _, s := range domain.Host {
		if ok := strings.Contains(host, s); ok {
			return true
		}
	}

	return false
}

// @Title SendToUser
// @Description 消息发送给用户
// @Param   userId  uint64   用户id
// @Param   code  int16  返回码
// @Param   message  string   消息
// @Param   method  string   方法路径
// @Param   data  interface   数据
// @Param   types  uint8  消息类型

func (b *BaseServer) SendToId(userId string, code int16, message string, method string, data interface{}, types uint8) {
	if userId == "" {
		zaplogger.Error("SendToUser--err->:", code, message, method, data, "user is not 0")
		return
	}
	b.SendToIds([]string{userId}, code, message, method, data, types)
}

// @Title SendToIds
// @Description 批量消息发送给用户群
// @Param   userIds  []uint64   用户ids
// @Param   code  int16  返回码
// @Param   message  string   消息
// @Param   method  string   方法路径
// @Param   data  interface   数据
// @Param   types  uint8  消息类型

func (b *BaseServer) SendToIds(userIds []string, code int16, message string, method string, data interface{}, types uint8) {
	var returnMessage model.SendMessage
	returnMessage.Type = types
	returnMessage.Message = message
	returnMessage.Code = code
	returnMessage.Data = data
	returnMessage.Method = method

	defer func() {
		if r := recover(); r != nil {
			zaplogger.Error("SendToIds panic：", r, returnMessage, userIds)
		}
	}()

	number := len(userIds)

	for i := 0; i < number; i++ { //looping from 0 to the length of the array
		userId := userIds[i]
		model.Clients.Range(func(key, value interface{}) bool {
			if key == userId {
				zaplogger.Info("SendToUsers-send-to->:", userId)
				conn, ok := value.(*websocket.Conn)
				if ok {
					err := conn.WriteJSON(returnMessage)
					if err != nil {
						zaplogger.Error("SendToUsers-send-to-user-err->:", returnMessage, err, userId)
					}
					zaplogger.Info("SendToUsers-send-to-user:", returnMessage, err, userId)
				} else {
					zaplogger.Error("SendToUsers--err->client is null:", returnMessage, "no user id", userId)
				}
			}
			return true
		})
	}
}

// @Title Converts
// @Description 遍历变量里所有数值到一个字符串数组
// @Param   dst  []string  原始数组
// @return   v reflect.Value  变量的reflect.ValueOf() 值
// @return   []string   字符串数组

func (b *BaseServer) Converts(dst []string, v reflect.Value) []string {
	// Drill down to the concrete value
	for v.Kind() == reflect.Interface {
		v = v.Elem()
	}

	if v.Kind() == reflect.Slice {
		// Convert each element of the slice.
		for i := 0; i < v.Len(); i++ {
			dst = b.Converts(dst, v.Index(i))
		}
	} else {
		// Convert value to string and append to result
		dst = append(dst, fmt.Sprint(v.Interface()))
	}
	return dst
}

// @Title WriteFile
// @Description 写入文件
// @Param   file  string  文件名
// @Param   content  string  文件内容

func (b *BaseServer) WriteFile(file string, content string) error {
	f, err := os.OpenFile(file, os.O_WRONLY, 0644)
	if err != nil {
		// 打开文件失败处理
		zaplogger.Error("打开域名文件失败.....", err)
		return err
	}

	str, err := b.ReadFile(file)
	if err != nil {
		return err
	}

	var domain model.Domain
	err = json.Unmarshal([]byte(str), &domain)
	if err != nil {
		return err
	}

	domain.Host = append(domain.Host, content)
	domainStr, err := json.Marshal(domain)

	if err != nil {
		return err
	}

	f.Write(domainStr)

	//// 查找文件末尾的偏移量
	//n, _ := f.Seek(0, 2)
	//// 从末尾的偏移量开始写入内容
	//_, err = f.WriteAt([]byte(content), n)

	defer f.Close()
	return nil
}

// @Title WriteFile
// @Description 获取文件
// @Param   file  string  文件名
// @Param   content  string  文件内容

func (b *BaseServer) ReadFile(file string) (string, error) {
	f, err := ioutil.ReadFile(file)
	if err != nil {
		return "", err
	}
	return string(f), nil
}
