<?php

namespace Server\sso;

class Sign
{
    protected $values = array();
    protected $signtype = 'MD5';
    protected $signkey = '';
    private static $PRIVATE_KEY = 'private_key.pem';
    private static $PUBLIC_KEY = 'public_key.pem';

    /**
     * 设置签名，详见签名生成算法类型
     * @param string $value
     **/
    public function SetSignType($sign_type)
    {
        $this->values['sign_type'] = $sign_type;
        $this->signtype = $sign_type;
        return $sign_type;
    }

    /**
     * 设置签名密钥
     * @param string $value
     **/
    public function SetSignKey($sign_key)
    {
        $this->signkey = $sign_key;
        return $sign_key;
    }
    /**
     * 设置签名，详见签名生成算法
     * @param string $value
     **/
    public function SetSign()
    {
        $sign = $this->MakeSign();
        $this->values['sign'] = $sign;
        return $sign;
    }

    /**
     * 获取签名，详见签名生成算法的值
     * @return 值
     **/
    public function GetSign()
    {
        return $this->values['sign'];
    }

    /**
     * 判断签名，详见签名生成算法是否存在
     * @return true 或 false
     **/
    public function IsSignSet()
    {
        return array_key_exists('sign', $this->values);
    }

    /**
     * 输出xml字符
     * @throws WxPayException
     **/
    public function ToXml()
    {
        if (!is_array($this->values) || count($this->values) <= 0) {
            exit("数组数据异常！");
        }

        $xml = "<xml>";
        foreach ($this->values as $key => $val) {
            if (is_numeric($val)) {
                $xml .= "<" . $key . ">" . $val . "</" . $key . ">";
            } else {
                $xml .= "<" . $key . "><![CDATA[" . $val . "]]></" . $key . ">";
            }
        }
        $xml .= "</xml>";
        return $xml;
    }

    /**
     * 将xml转为array
     * @param string $xml
     * @throws WxPayException
     */
    public function FromXml($xml)
    {
        if (!$xml) {
            exit("xml数据异常！");
        }
        //将XML转为array
        //禁止引用外部xml实体
        libxml_disable_entity_loader(true);
        $this->values = json_decode(json_encode(simplexml_load_string($xml, 'SimpleXMLElement', LIBXML_NOCDATA)), true);
        return $this->values;
    }

    /**
     *
     * 使用数组初始化
     * @param array $array
     */
    public function FromArray($array)
    {
        $this->values = $array;
    }

    /**
     * 格式化参数格式化成url参数
     */
    public function ToUrlParams()
    {
        $buff = "";
        foreach ($this->values as $k => $v) {
            if ($k != "sign" && $v != "" && !is_array($v)) {
                $buff .= $k . "=" . $v . "&";
            }
        }

        $buff = trim($buff, "&");
        return $buff;
    }

    /**
     * 生成签名
     * @param bool $needSignType 是否需要补signtype
     * @return 签名，本函数不覆盖sign成员变量，如要设置签名需要调用SetSign方法赋值
     */
    public function MakeSign($needSignType = true)
    {
        if ($needSignType) {
            $this->SetSignType($this->signtype);
        }
        //签名步骤一：按字典序排序参数
        ksort($this->values);
        foreach ($this->values as $k => $v) {
            if (is_array($v)) {
                $this->values[$k] = json_encode($v);
            }
        }
        $string = $this->ToUrlParams();
        //签名步骤二：在string后加入KEY
        $string = $string . "&key=" . $this->signkey;
        //签名步骤三：MD5加密或者HMAC-SHA256
        if ($this->signtype == "MD5") {
            $string = md5($string);
            $result = strtoupper($string);
        } else if ($this->signtype == "HMAC-SHA256") {
            $string = hash_hmac("sha256", $string, $this->signkey);
            $result = strtoupper($string);
        } else if ($this->signtype == "Rsa2") {
            $result = $this->createSign($string);
        } else {
            exit("签名类型不支持！");
        }

        //签名步骤四：所有字符转为大写
        //$result = strtoupper($string);
        return $result;
    }

    /**
     * 获取设置的值
     */
    public function GetValues()
    {
        return $this->values;
    }

    /**
     * 检测签名
     */
    public function CheckSign()
    {
        if (!$this->IsSignSet()) {
            return false;
        }
        if ($this->signtype == "Rsa2") {
            if (empty($this->values)) {
                return false;
            }
            ksort($this->values);
            foreach ($this->values as $k => $v) {
                if (is_array($v)) {
                    $this->values[$k] = json_encode($v);
                }
            }
            $string = $this->ToUrlParams();
            $string = $string . "&key=" . $this->signkey;
            if ($this->verifySign($string, $this->values['sign']) === true) {
                //签名正确
                return true;
            }
        } else {
            $sign = $this->MakeSign();
            if ($this->GetSign() == $sign) {
                //签名正确
                return true;
            }
        }
        return false;
    }

    /**
     * 获取私钥
     * @return bool|resource
     */
    private static function getPrivateKey()
    {
        if (file_exists(self::$PRIVATE_KEY)) {
            $privateKey = file_get_contents(self::$PRIVATE_KEY);
            return openssl_pkey_get_private($privateKey);
        } else {
            return null;
        }
    }

    /**
     * 获取公钥
     * @return bool|resource
     */
    private static function getPublicKey()
    {
        if (file_exists(self::$PUBLIC_KEY)) {
            $publicKey = file_get_contents(self::$PUBLIC_KEY);
            return openssl_pkey_get_public($publicKey);
        } else {
            return null;
        }
    }

    /**
     * 私钥加密
     * @param string $data
     * @return null|string
     */
    public static function privateEncrypt($data = '')
    {
        if (!is_string($data)) {
            return null;
        }
        return openssl_private_encrypt($data, $encrypted, self::getPrivateKey()) ? base64_encode($encrypted) : null;
    }

    /**
     * 公钥加密
     * @param string $data
     * @return null|string
     */
    public static function publicEncrypt($data = '')
    {
        if (!is_string($data)) {
            return null;
        }
        return openssl_public_encrypt($data, $encrypted, self::getPublicKey()) ? base64_encode($encrypted) : null;
    }

    /**
     * 私钥解密
     * @param string $encrypted
     * @return null
     */
    public static function privateDecrypt($encrypted = '')
    {
        if (!is_string($encrypted)) {
            return null;
        }
        return (openssl_private_decrypt(base64_decode($encrypted), $decrypted, self::getPrivateKey())) ? $decrypted : null;
    }

    /**
     * 公钥解密
     * @param string $encrypted
     * @return null
     */
    public static function publicDecrypt($encrypted = '')
    {
        if (!is_string($encrypted)) {
            return null;
        }
        return (openssl_public_decrypt(base64_decode($encrypted), $decrypted, self::getPublicKey())) ? $decrypted : null;
    }

    /**
     * 创建签名
     * @param string $data 数据
     * @return null|string
     */
    public function createSign($data = '')
    {
        if (!is_string($data)) {
            return null;
        }
        return openssl_sign($data, $sign, self::getPrivateKey(), OPENSSL_ALGO_SHA256) ? base64_encode($sign) : null;
    }

    /**
     * 验证签名
     * @param string $data 数据
     * @param string $sign 签名
     * @return bool
     */
    public function verifySign($data = '', $sign = '')
    {
        if (!is_string($sign) || !is_string($sign)) {
            return false;
        }
        return (bool)openssl_verify($data, base64_decode($sign), self::getPublicKey(), OPENSSL_ALGO_SHA256);
    }

    public function getParams($url = '')
    {
        $refer_url = parse_url($url);
        $params = $refer_url['query'];
        $arr = array();
        if (!empty($params)) {
            $paramsArr = explode('&', $params);
            foreach ($paramsArr as $k => $v) {
                $a = explode('=', $v);
                $arr[$a[0]] = $a[1];
            }
        }
        return $arr;
    }
}
