<?php

namespace Server\ucenter;

use App\Mail\VerifyCode;
use App\Services\HttpService;
use App\Services\MemberService;
use App\Services\MicroService;
use Server\ucenter\UcenterWiki;
use Illuminate\Support\Facades\Mail;

class UcenterService extends MicroService {

    public $servername = 'ucenter';
    public $uniacid = 0;
    public $REGULAR_EMAIL = '/\w+([-+.]\w+)*@\w+([-.]\w+)*\.\w+([-.]\w+)*/i';
    public $REGULAR_MOBILE = '/1[3456789][0-9]{9}/';
    public $authkey = '';
    public $setting = [];
    public $defaultAvatar = "avatar.png";
    public $defaultGroup = [];
    public $globalAvatar = [];
    public $credit_types = array('credit1','credit2','credit3','credit4','credit5','credit6');
    public $tables = array(
        'member'=>'mc_members',
        'group'=>'mc_groups',
        'creditrecord'=>'mc_credits_record'
    );

    public function __construct($uniacid=0){
        parent::__construct($this->servername);
        global $_W;
        if (empty($uniacid)){
            $uniacid = $_W['uniacid'];
        }
        $this->uniacid = intval($uniacid);
        $this->Unique = true;
        $this->getSetting();
        if (isset($_W['config']['setting']['authkey'])){
            $this->authkey = $_W['config']['setting']['authkey'];
        }
        $defaultGroup = pdo_get($this->tables['group'], array('uniacid'=>$this->uniacid,'isdefault'=>1));
        if (empty($defaultGroup)){
            $defaultGroup = array("uniacid"=>$this->uniacid, "title"=>"默认用户组", "credit"=>0, "price"=>0.00, "paytype"=>0, "isdefault"=>1);
            $groupid = pdo_insertgetid($this->tables['group'], $defaultGroup);
            $defaultGroup['groupid'] = $groupid;
            $this->setting['groups'][$groupid] = $defaultGroup;
        }
        $this->defaultGroup = $defaultGroup;
    }

    /**
     * HTTP方式访问
     * @param string|null $platform 路由通道，可选web、app、api及自定义通道
     * @param string|null $route 路由名称
     * @return array|\error 返回接口数据或报错信息
     * @throws \Exception
     */
    public function HttpRequest($platform="web", $route=""){
        global $_GPC, $_W;
        if (empty($route)){
            $route = empty($_GPC['ctrl']) ? 'index' : trim($_GPC['ctrl']);
        }
        $route = str_replace(".","/",$route);
        list($controller, $method) = explode("/", $route);
        if (empty($method)) $method = 'main';

        $class = "Server\\{$this->identity}\\{$platform}\\" . ucfirst($controller) . "Controller";
        if (!class_exists($class)){
            $class = "Server\\{$this->identity}\\{$platform}\IndexController";
            $method = $controller;
            $controller = 'index';
        }

        $_W['controller'] = $controller;
        $_W['action'] = $method;

        if (!class_exists($class)) return error(-1,__('controllerNotFound', ['ctrl'=>$class]));
        $instance = new $class();
        if (!method_exists($instance,$method)) return error(-1,"Method $class::$method() dose not exist!");
        return $instance->$method();
    }

    public function Processor($listener, $data = array()){
        switch ($listener){
            case "storage.remove" : {
                //检测用户头像是否被删除
                $uid = (int)pdo_getcolumn($this->tables['member'], array('avatar'=>$data['path']), "uid");
                if ($uid){
                    $this->UpdateInfo($uid, array('avatar'=>$this->defaultAvatar));
                }
                break;
            }
            default :
        }
        return parent::Processor($listener, $data); // TODO: Change the autogenerated stub
    }

    public function avatar($uid, $default=''){
        if ($default) $avatar = $default;
        if(empty($avatar)){
            if (isset($this->globalAvatar[$uid])){
                $avatar = $this->globalAvatar[$uid];
            }else{
                $avatar = pdo_getcolumn($this->tables['member'], array('uid'=>intval($uid),'uniacid'=>$this->uniacid),'avatar');
            }
        }
        $this->globalAvatar[$uid] = $avatar;
        return $avatar ? tomedia($avatar) : $this->res("avatar.png");
    }

    public function getSetting($key=""){
        if(empty($this->setting)){
            global $_W;
            $setting = $this->SettingLoad(array('grouplevel','creditbehaviors','passport','oauth','uc', 'ucsync'), $this->uniacid);
            $setting['groups'] = $_W['account']['groups'];
            $setting['grouplevel'] = intval($setting['grouplevel']);
            if (empty($setting['creditbehaviors'])){
                $setting['creditbehaviors'] = array(
                    'currency'=>'credit2',
                    'activity'=>'credit1'
                );
            }
            if (empty($setting['ucsync'])){
                $setting['ucsync'] = array(
                    'sync_user'=>false,
                    'sync_credit'=>false,
                    'sync_user_url'=>"",
                    'sync_credit_url'=>""
                );
            }
            $creditNames = $this->SettingLoad('creditnames', $this->uniacid);
            $this->setting = array_merge($setting, $creditNames);
        }
        if (empty($key)) return $this->setting;
        return $this->setting[$key];
    }

    public function getUser($uid, $fields=array(), $field="uid"){
        $condition = array('uniacid'=>$this->uniacid);
        $condition[$field] = $uid;
        $member = pdo_get($this->tables['member'], $condition, $fields);
        if (empty($member)) return [];
        if (empty($member['nickname'])) $member['nickname'] = "用户" . $member['uid'];
        return $member;
    }

    public function getUserCredit($uid, $types = array()){
        if (empty($types) || $types == '*') {
            $select = array('credit1', 'credit2', 'credit3', 'credit4', 'credit5', 'credit6');
        } else {
            $struct = $this->credit_types;
            foreach ($types as $key => $type) {
                if (!in_array($type, $struct)) {
                    unset($types[$key]);
                }
            }
            $select = $types;
        }
        return pdo_get('mc_members', array('uid' => $uid, 'uniacid'=>$this->uniacid), $select);
    }

    public function getUsers($condition=array(),$page=1,$fetch='all'){
        $where = "uniacid=".$this->uniacid;
        $params = array();
        if ($condition['groupid']){
            $where .= " and groupid=:groupid";
            $params[':groupid'] = intval($condition['groupid']);
        }
        if(!empty($condition['keyword'])){
            $params[':keyword'] = "%".trim($condition['keyword'])."%";
            $params[':keytext'] = trim($condition['keyword']);
            $where .= " and (nickname like :keyword or mobile=:keytext or uid=:keytext or email like :keyword)";
        }
        $order = " order by createtime desc";
        $pagesize = empty($condition['pagesize']) ? 15 : intval($condition['pagesize']);
        $list = [];
        if ($fetch=='list' || $fetch=='all'){
            $limit = "";
            if ($page!=-1){
                $page = max(1,intval($page));
                $limit = " limit ".($page-1)*$pagesize .",$pagesize";
            }
            $select = empty($condition['select']) ? "*" : trim($condition['select']);
            $list = pdo_fetchall("select $select from ".tablename($this->tables['member'])." where ".$where.$order.$limit,$params);
            if(!empty($list)){
                $groups = $this->setting['groups'];
                foreach ($list as &$value){
                    $value['avatar'] = $this->avatar($value['uid'],$value['avatar']);
                    if (!isset($groups[$value['groupid']])){
                        $value['groupid'] = $this->defaultGroup['groupid'];
                    }
                    $value['grouptitle'] = $groups[$value['groupid']]['title'];
                    $value['datetime'] = date('Y-m-d H:i', $value['createtime']);
                    if(strexists($value['email'],'@qingruyun.com') || strexists($value['email'], '@we7.cc')){
                        $value['email'] = "";
                    }
                }
            }
            if($fetch=='list') return $list;
        }
        if ($fetch=='total' || $fetch=='all'){
            $total = (int)pdo_fetchcolumn("select count(*) from ".tablename($this->tables['member'])." where ".$where,$params);
            if($fetch=='total') return $total;
        }
        $pager = pagination($total, $page, $pagesize);
        return array($list, $total, $pager);
    }

    public function getGroup($groupid){
        $groupid = is_array($groupid) ? $groupid['groupid'] : intval($groupid);
        return pdo_get($this->tables['group'],array('uniacid'=>$this->uniacid,'groupid'=>$groupid));
    }

    public function getCreditRecord($condition=array(),$page=1,$fetch='all'){
        $credittype = $condition['type'] ?: 'credit2';
        $where = ' WHERE uniacid = :uniacid AND credittype = :credittype';
        $params = array(':uniacid' => $this->uniacid, ':credittype' => $credittype);
        if ($condition['starttime']){
            $where .= ' AND createtime >= :starttime';
            $params[':starttime'] = $condition['starttime'];
        }
        if ($condition['endtime']){
            $where .= ' AND createtime <= :endtime';
            $params[':endtime'] = $condition['endtime'];
        }
        if ($condition['num']){
            if (intval($condition['num'])==1){
                $where .= ' AND num >= 0';
            }elseif (intval($condition['num'])>0){
                $where .= ' AND num <= 0';
            }
        }
        if ($condition['min']){
            $where .= ' AND abs(num) >= :minnum';
            $params[':minnum'] = intval($condition['min']);
        }
        if ($condition['max']){
            $where .= ' AND abs(num) <= :maxnum';
            $params[':maxnum'] = intval($condition['max']);
        }
        if ($condition['uid']){
            $where .= ' AND uid='.intval($condition['uid']);
        }
        if ($condition['remark']){
            $where .= ' AND remark like :remark';
            $params[':remark'] = "%".trim($condition['remark'])."%";
        }
        if ($condition['keyword']){
            $keyword = "%".trim($condition['keyword'])."%";
            $where .= ' AND (uid IN (SELECT uid FROM '.tablename($this->uniacid).' WHERE uniacid = :uniacid AND (nickname LIKE '.$keyword.' OR uid = :userid OR mobile LIKE '.$keyword.')))';
            $params[':userid'] = intval($condition['keyword']);
        }
        $psize = $condition['pagesize'] ?: 15;
        if ($fetch=='list' || $fetch=='all'){
            if ($page>=0){
                $pindex = max(1, intval($page));
                $limit = " ORDER BY id DESC LIMIT " . ($pindex - 1) * $psize . ", {$psize}";
            }else{
                $limit = '';
            }
            $list = pdo_fetchall('SELECT * FROM ' . tablename($this->tables['creditrecord']) . $where . $limit, $params);
            if ($list){
                $members = array();
                foreach ($list as &$value){
                    $member = isset($members[$value['uid']]) ? $members[$value['uid']] : $this->getUser($value['uid'], array('nickname','avatar','realname','mobile'));
                    $value['nickname'] = $member ? $member['nickname'] : '未知用户';
                    $value['avatar'] = $this->avatar($value['uid'], $member['avatar']);
                    $value['realname'] = $member ? $member['realname'] : '未知用户';
                    $value['mobile'] = $member ? $member['mobile'] : '未知';
                    $value['createdate'] = date('Y-m-d H:i',$value['createtime']);
                    if (!isset($members[$value['uid']])) $members[$value['uid']] = $member;
                }
            }
            if ($fetch=='list') return $list;
        }
        if ($fetch=='total' || $fetch=='all'){
            $total = (int)pdo_fetchcolumn('SELECT COUNT(*) FROM ' . tablename($this->tables['creditrecord']) . $where, $params);
            if ($fetch=='total') return $total;
        }
        if ($fetch=='sum' || $fetch=='all'){
            $sum = (float)pdo_fetchcolumn('SELECT SUM(ABS(num)) FROM ' . tablename($this->tables['creditrecord']) . $where, $params);
            if ($fetch=='sum') return $sum;
        }
        if ($fetch=='increment' || $fetch=='all'){
            $increment = (float)pdo_fetchcolumn('SELECT SUM(num) FROM ' . tablename($this->tables['creditrecord']) . $where." and num>0", $params);
            if ($fetch=='increment') return $increment;
        }
        if ($fetch=='decrement' || $fetch=='all'){
            $decrement = (float)pdo_fetchcolumn('SELECT SUM(ABS(num)) FROM ' . tablename($this->tables['creditrecord']) . $where." and num<0", $params);
            if ($fetch=='decrement') return $decrement;
        }
        $pg = pagination($total, $pindex, $psize);
        return array($list,$total,$pg, $sum, $increment, $decrement);
    }

    public function getGroupDefault($fields=array()){
        return pdo_get($this->tables['group'],array('uniacid'=>$this->uniacid,'isdefault'=>1), $fields);
    }

    public function removeGroup($groupid){
        $group = $this->getGroup($groupid);
        if(!$group) return true;
        if($group['isdefault']) return false;
        $complete = pdo_delete($this->tables['group'],array('groupid'=>$group['groupid']));
        if ($complete){
            $default = $this->getGroupDefault(['groupid']);
            pdo_update($this->tables['member'], array('groupid'=>$default['groupid']), array('groupid'=>$group['groupid'],'uniacid'=>$this->uniacid));
        }
        return $complete;
    }

    public function SetGroupDefault(){
        $default = pdo_fetch("select groupid from ".t('mc_groups')." where uniacid=".$this->uniacid." order by credit asc,price asc limit 1");
        return $this->UpdateGroup(array('isdefault'=>1), $default['groupid']);
    }

    public function UpdateGroupId($uid){
        if ($this->setting['grouplevel']>0){
            $member = $this->getUser($uid, array('uid','credit1','groupid'));
            if (empty($member)) return false;
            $groupforward = pdo_fetch("select * from ".tablename($this->tables['group'])." where credit>=:credit and uniacid=:uniacid order by credit asc limit 1", array(":credit"=>$member['credit'], "uniacid"=>$this->uniacid));
            if ($groupforward['groupid']==$member['groupid']) return true;
            $groupnow = $this->getGroup($member['groupid']);
            //根据积分只升不降
            if ($this->setting['grouplevel']==2 && $groupnow['credit']>$groupforward['credit']){
                return true;
            }
            //根据积分自动升降
            $this->UpdateInfo($member['uid'], array('groupid'=>$groupforward['groupid']));
        }
        return true;
    }

    public function UpdateGroup($data, $groupid=0){
        if (!empty($groupid)){
            $condition = is_array($groupid) ? $groupid : array('groupid'=>intval($groupid));
            $condition['uniacid'] = $this->uniacid;
            return pdo_update($this->tables['group'], $data, $condition);
        }else{
            $data['uniacid'] = $this->uniacid;
            return pdo_insertgetid($this->tables['group'], $data);
        }
    }

    public function getMethods($data=array()){
        return parent::getMethods(UcenterWiki::Methods($data));
    }

    public function getToken($uid,$password='',$salt='', $expire = 60){
        $expireAt = TIMESTAMP + $expire;
        return base64_encode(json_encode(array('uid'=>$uid,'expire'=>$expireAt,'hash'=>md5($password.$salt.$expireAt))));
    }

    /**
     * 发送邮件
     * @param string $to 发送者
     * @param string $code 验证码
     * @param string|null $name 平台名称
    */
    public function sendEmailCode($to, $code, $name=""){
        if (defined("QingFrame")){
            try {
                if (!Mail::to($to)->send(new VerifyCode($code))){
                    return error(-1, "发送失败，请重试");
                }
                return true;
            }catch (\Exception $exception){
                return error(-1, $exception->getMessage());
            }
        }
        if (empty($name)){
            $name = $GLOBALS['_W']['account']['name'];
        }
        load()->func('communication');
        return ihttp_email($to,  "【{$name}】平台验证码", "您的验证码是：{$code}，15分钟内有效。祝您生活愉快！【{$name}】");
    }

    /**
     * 唯一性检测
     * @param string $value 检测词
     * @param string|null $field 检查字段
     * @return bool|int 查询结果，如存在则返回UID
     */
    public function CheckUni($value, $field='mobile'){
        if (empty($field) || empty($value)) return false;
        $data = array('uniacid'=>$this->uniacid);
        $data[$field] = trim($value);
        return pdo_getcolumn($this->tables['member'], $data, 'uid');
    }

    /**
     * 操作积分
     * @param int $uid 用户UID
     * @param string|null $type 积分类型（credit1、credit2、credit3、credit4、credit5、credit6）
     * @param float|int|null $count 操作金额，正数为加负数为减
     * @param array|string|null $params 额外参数，包括等remark（备注）、store_id（门店ID）、clerk_id（店员ID）等
     * @return array|bool 修改结果
     */
    public function UpdateCredit($uid, $type='credit2', $count=0, $params=array()){
        if (!in_array($type, $this->credit_types)) {
            return error('-1', "无效的积分类型 {$type}");
        }
        if (empty($count)) {
            return true;
        }
        if(empty($uid)){
            return error(-1,"无效的用户UID");
        }
        $value = pdo_getcolumn($this->tables['member'], array('uid'=>intval($uid),'uniacid'=>$this->uniacid), $type);
        $creditval = floatval($count);
        if($creditval<0 && ($value+$creditval)<0){
            return error('-1', "对应的积分{$type}不足");
        }

        // 计算变动后的余额
        $balance_after = $value + $creditval;

        if (!$this->UpdateInfo($uid,array($type=>$balance_after))){
            return error(-1,'操作失败，请重试');
        }
        $this->UpdateGroupId($uid);
        $data = array(
            'uid' => $uid,
            'credittype' => $type,
            'uniacid' => $this->uniacid,
            'num' => $creditval,
            'balance_after' => $balance_after,
            'createtime' => TIMESTAMP,
            'operator' => intval($params['operator']),
            'module' => MODULE_IDENTIFIE,
            'clerk_id' => intval($params['clerk_id']),
            'store_id' => intval($params['store_id']),
            'clerk_type' => isset($params['clerk_type']) ? intval($params['clerk_type']) : 1,
            'remark' => is_array($params) ? $params['remark'] : trim($params),
            'real_uniacid' => $this->uniacid
        );
        pdo_insert($this->tables['creditrecord'], $data);
        if ($this->setting['ucsync']['sync_credit'] && !empty($this->setting['ucsync']['sync_credit_url'])){
            $syncData = array(
                'uid'=>$uid,
                'type'=>$type,
                'uniacid'=>$this->uniacid,
                'amount'=>$creditval,
                'operator'=>$data['operator'],
                'module'=> defined("MODULE_IDENTIFIE")?MODULE_IDENTIFIE:"",
                'remark'=>$data['remark'],
                'createTime'=>TIMESTAMP,
                'store_id'=>$data['store_id']
            );
            if (defined('QingFrame')){
                HttpService::ihttp_post($this->setting['ucsync']['sync_credit_url'], $syncData);
            }else{
                load()->func('communication');
                ihttp_post($this->setting['ucsync']['sync_credit_url'], $syncData);
            }
        }
        return true;
    }

    /**
     * 更新资料
     * @param int $uid UID
     * @param array $profile 用户资料数组，字段一一对应
     * @return array|bool
     */
    public function UpdateInfo($uid, $data){
        //检测手机号
        if (!empty($data['mobile'])){
            //合法性
            if (!preg_match($this->REGULAR_MOBILE, $data['mobile'])){
                return $this->error("手机号格式不正确");
            }
            //唯一性
            $mobileexists = $this->CheckUni($data['mobile']);
            if (!empty($mobileexists) && $uid!=$mobileexists){
                return $this->error("该手机号已被注册");
            }
        }
        //检测邮箱
        if (!empty($data['email'])){
            //合法性
            if (!preg_match($this->REGULAR_EMAIL, $data['email'])){
                return $this->error("邮箱格式不正确");
            }
            //唯一性
            $userexists = $this->CheckUni($data['email'],'email');
            if (!empty($userexists) && $uid!=$userexists){
                return $this->error("该邮箱已被注册");
            }
        }
        $complete = pdo_update($this->tables['member'], $data, array('uid'=>intval($uid), 'uniacid'=>$this->uniacid));
        if (!$complete) return $this->error("保存失败，请重试");
        if (isset($data['credit1'])){
            $this->UpdateGroupId($uid);
        }
        return true;
    }

    /**
     * 修改密码
     * @param int $uid 用户UID
     * @param string $password 新密码
     * @param string|null $salt 加密盐
     * @return array|bool 修改结果
     */
    public function ResetPassword($uid, $password, $salt=""){
        //验证密码合法性，待完善
        $data = array('salt'=>trim($salt));
        if (empty($salt)){
            $data['salt'] = random(8);
        }
        $data['password'] = md5($password . $data['salt'] . $this->authkey);
        if (!pdo_update($this->tables['member'],$data, array('uid'=>intval($uid), 'uniacid'=>$this->uniacid))){
            return $this->error("更新失败，请重试");
        }
        return true;
    }

    /**
     * 统一用户注册
     * @param string $username 用户名
     * @param string|null 登录密码
     * @param array|null $profile 用户信息
     * @return array 用户信息
    */
    public function register($username,$password='',$profile=array()){
        $condition = array('uniacid'=>$this->uniacid);
        $username = trim($username);
        if (empty($password)){
            $password = random(12);
        }
        if(preg_match($this->REGULAR_EMAIL, $username)){
            $condition['email'] = $username;
            $type = 'email';
        }elseif (preg_match($this->REGULAR_MOBILE, $username)){
            $condition['mobile'] = $username;
            $type = 'mobile';
        }else{
            return error(-1,"用户名格式不正确");
        }
        $user = pdo_get($this->tables['member'],$condition);
        if (!empty($user)) return error(-5,'用户'.$username.'已注册，请直接登录。');
        $data = $profile;
        $data['salt'] = $data['salt'] ?: random(8);
        if (empty($data['groupid'])){
            $defaultgroup = $this->getGroupDefault(array('groupid'));
            $data['groupid'] = $defaultgroup['groupid'];
        }
        $data['email'] = $type == 'email'  ? $username : '';
        $data['mobile'] = $type == 'mobile' ? $username : '';
        $data['password'] = md5($password . $data['salt'] . $this->authkey);
        $data['uniacid'] = $this->uniacid;
        $data['createtime'] = TIMESTAMP;
        if (!$data['nickname']) $data['nickname'] = '用户'.substr(TIMESTAMP.'',4);
        $uid = pdo_insertgetid('mc_members',$data);
        if (!$uid){
            return error(-1,'注册失败，请重试。');
        }
        $data['uid'] = $uid;
        if ($this->setting['ucsync']['sync_user'] && !empty($this->setting['ucsync']['sync_user_url'])){
            //同步到第三方
            $syncData = $data;
            $syncData['username'] = $username;
            unset($syncData['password'], $syncData['salt']);
            if (defined('QingFrame')){
                HttpService::ihttp_post($this->setting['ucsync']['sync_user_url'], $syncData);
            }else{
                load()->func('communication');
                ihttp_post($this->setting['ucsync']['sync_user_url'], $syncData);
            }
        }
        $this->Event("uc.register", $data);
        return $data;
    }

    /**
     * 账户注销
     * @param int $uid 用户UID
     * @return bool 注销结果
     */
    public function cancellation($uid){
        if (pdo_delete($this->tables['member'], array('uniacid'=>$this->uniacid,'uid'=>intval($uid)))){
            $this->Event("uc.cancellation", array('uid'=>intval($uid))); //事件广播
            return true;
        }
        return false;
    }

    /**
     * 用户选择器
     * @param string|null $name 输入框字段名(name)
     * @param int|null $value 当前选中用户UID
     * @param array|null $options 输入框属性，如placeholder、required等
     * @return string 表单项HTML
     */
    public function tpl_form_user($name, $value='', $options=array()){
        $user = array('nickname'=>"",'avatar'=>$this->defaultAvatar,'uid'=>intval($value));
        $pickerID = 'mp'.random(8,true);
        $pickerUrl = $this->url("member/picker");
        if (!empty($options['url'])){
            $pickerUrl = $options['url'];
        }
        $selectItem = "";
        if (!empty($user['uid'])){
            $user = $this->getUser($user['uid'], array('uid','nickname','avatar'));
            $user['avatar'] = $this->avatar($user['uid'], $user['avatar']);
            $selectItem = '<dd data-uid="'.$user['uid'].'" data-avatar="'.$user['avatar'].'" class="layui-select-tips layui-this">'.$user['nickname'].'</dd>';
        }
        if (empty($options['placeholder'])){
            $options['placeholder'] = '输入UID/昵称/手机号搜索';
        }
        $extraattr = ' placeholder="'.$options['placeholder'].'"';
        if (!empty($options['required'])){
            $extraattr .= ' required lay-verify="required"';
        }
        $html = '<div class="layui-input-block member-picker" data-pid="'.$pickerID.'" data-url="'.$pickerUrl.'">
                <input type="hidden" name="'.$name.'" value="'.$user['uid'].'" id="'.$pickerID.'">
                <div class="layui-form-select">
                    <div class="layui-select-title">
                        <input type="text" id="'.$pickerID.'-input" name="memberpicker[]" value="'.$user['nickname'].'" autocomplete="off" class="layui-input"'.$extraattr.' />
                        <i class="layui-edge"></i>
                        <i class="layui-icon layui-icon-close"></i>
                    </div>
                    <dl class="layui-anim layui-anim-upbit">
                        <dd data-uid=""'.(empty($user['uid'])?' class="layui-select-tips layui-this"':'').'>输入UID/昵称/手机号搜索</dd>
                        '.$selectItem.'
                    </dl>
                </div>
            </div>';
        if (!empty($options['preview'])){
            $html .= '<div class="layui-input-block input-group" style="margin-top:.5em;">
                <img id="'.$pickerID.'-avatar" src="'.$user['avatar'].'" data-val="'.$this->defaultAvatar.'" class="img-responsive img-thumbnail" width="132"  alt="'.$user['nickname'].'"/>
            </div>';
        }
        if (!defined("UCENTER_TPL_USER")){
            $html .= '<script type="text/javascript">$(function (){ Core.MemberPicker(); })</script>';
            define("UCENTER_TPL_USER", true);
        }
        return $html;
    }

}
